// LICENSE
// =======
// Copyright (c) 2017-2019 Advanced Micro Devices, Inc. All rights reserved.
// -------
// Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated documentation
// files (the "Software"), to deal in the Software without restriction, including without limitation the rights to use, copy,
// modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
// -------
// The above copyright notice and this permission notice shall be included in all copies or substantial portions of the
// Software.
// -------
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE AUTHORS OR
// COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE

//Initial port to ReShade: SLSNe    https://gist.github.com/SLSNe/bbaf2d77db0b2a2a0755df581b3cf00c
//Optimizations by Marty McFly:
//     vectorized math, even with scalar gcn hardware this should work
//     out the same, order of operations has not changed
//     For some reason, it went from 64 to 48 instructions, a lot of MOV gone
//     Also modified the way the final window is calculated
//      
//     reordered min() and max() operations, from 11 down to 9 registers    
//
//     restructured final weighting, 49 -> 48 instructions
//
//     delayed RCP to replace SQRT with RSQRT
//
//     removed the saturate() from the control var as it is clamped
//     by UI manager already, 48 -> 47 instructions
//
//
// PCSX2 Port by guest.r

#ifdef SHADER_MODEL // make safe to include in resource file to enforce dependency

#define Sharpness 0.7  // Filter sharpness, from 0.0 to 1.0

Texture2D Texture;
SamplerState TextureSampler;

struct PS_INPUT
{
	float4 p : SV_Position;
	float2 t : TEXCOORD0;
};

struct PS_OUTPUT
{
	float4 c : SV_Target0;
};

cbuffer cb0
{
    float2 _xyFrame;
    float4 _rcpFrame;
};

static float2 ps = _rcpFrame.xy;

PS_OUTPUT ps_main(PS_INPUT input)
{
	PS_OUTPUT output;
	float2 pos = input.t;	
	
	float2 dx = float2(ps.x, 0.0);
	float2 dy = float2(0.0, ps.y);
	float2 dg1 = dx + dy;
	float2 dg2 = dy - dx;
	
	float3 a = Texture.Sample(TextureSampler, pos - dg1).xyz; 
	float3 b = Texture.Sample(TextureSampler, pos - dy).xyz; 
	float3 c = Texture.Sample(TextureSampler, pos + dg2).xyz; 
	float3 d = Texture.Sample(TextureSampler, pos - dx).xyz; 
	float3 e = Texture.Sample(TextureSampler, pos).xyz; 
	float3 f = Texture.Sample(TextureSampler, pos + dx).xyz; 
	float3 g = Texture.Sample(TextureSampler, pos - dg2).xyz; 
	float3 h = Texture.Sample(TextureSampler, pos + dy).xyz; 
	float3 i = Texture.Sample(TextureSampler, pos + dg1).xyz;
	
	// Soft min and max.
	//  a b c             b
	//  d e f * 0.5  +  d e f * 0.5
	//  g h i             h
    // These are 2.0x bigger (factored out the extra multiply).
    float3 mnRGB = min(min(min(d, e), min(f, b)), h);
    float3 mnRGB2 = min(mnRGB, min(min(a, c), min(g, i)));
    mnRGB += mnRGB2;

    float3 mxRGB = max(max(max(d, e), max(f, b)), h);
    float3 mxRGB2 = max(mxRGB, max(max(a, c), max(g, i)));
    mxRGB += mxRGB2;

    // Smooth minimum distance to signal limit divided by smooth max.
    float3 rcpMRGB = rcp(mxRGB);
    float3 ampRGB = saturate(min(mnRGB, 2.0 - mxRGB) * rcpMRGB);    
    
    // Shaping amount of sharpening.
    ampRGB = rsqrt(ampRGB);

    float peak = 8.0 - 3.0 * Sharpness;
    float3 wRGB = -rcp(ampRGB * peak);

    float3 rcpWeightRGB = rcp(1.0 + 4.0 * wRGB);

    //                          0 w 0
    //  Filter shape:           w 1 w
    //                          0 w 0 
    float3 window = (b + d) + (f + h);
    float3 outColor = saturate((window * wRGB + e) * rcpWeightRGB); 
	
	output.c.a = 1.0;
	output.c.xyz = outColor;
	return output;
}

#endif
